#! /usr/bin/env python3
# coding: UTF-8
# entrypoint script to execute from CLI.

"""
Splits a map into inner and outer portions based on a given coordinate range

Given an SBYW map as input, two outputs will be produced.
A version of the complete map lacking all objects created inside the enclosing zone,
and all the objects in that enclosing zone.

One of the options is a mode switch, either enclosed (default) or overlapped.
If enclosed, only objects that are entirely enclosed within the given range, that is they have no minimums or maximums outside it, will be filtered.
If overlap, any objects that intersect the given range will be filtered.
Caution! If you build larger enclosing blocks and carve them out, overlap mode will destroy the enclosing blocks!

An aditional feature is to offset the filtered data by constants along each axis.
"""

import argparse
from pathlib import Path
import sys
from .functions import deltacoord, splitfilter
from .objects import parse_map

version = 2.2


# modified to override the exit method with one that respects the interactive mode
class ModifiedArgumentParser(argparse.ArgumentParser):
    def __init__(self, **kwargs):
        self.interactive = kwargs.pop("interactive", False)
        super().__init__(**kwargs)

    def exit(self, status=0, message=None):
        try:
            super().exit(status, message)
        except SystemExit as ex:  # this is raised by the original method
            if self.interactive:
                input("Press enter to exit")  # discard input
            raise ex  # reraise so it still exits

    def error(self, message):
        if not self.interactive:
            self.print_usage(sys.stderr)
        args = {"prog": self.prog, "message": message}
        self.exit(2, ("%(prog)s: error: %(message)s\n") % args)


def main():
    interactive = False
    if len(sys.argv) == 1:
        interactive = True
        try:
            interact()  # mutates sys.argv
        except Exception as exc:
            print(str(exc))
            input("Press enter to exit")
            sys.exit(1)
    else:
        print(f"CoordSlice version {version}, by x0")
    parser = ModifiedArgumentParser(
        prog="CoordSlice",
        description="Split an SBYW map into two parts based on an enclosing zone",
        interactive=interactive,
    )
    parser.add_argument(
        "--version", action="version", version=f"%(prog)s version {version}"
    )
    parser.add_argument(
        "infile", type=Path, metavar="input file", help="map file to read"
    )
    parser.add_argument(
        "outmap",
        type=Path,
        metavar="map output file",
        help="file to write all map lines but the filtered objects to",
    )
    parser.add_argument(
        "outfiltered",
        type=Path,
        metavar="filtered output file",
        help="file to write the filtered objects to",
    )
    parser.add_argument(
        "--overlap",
        dest="enclose",
        action="store_false",
        help="Filter objects which overlap the given range, rather than those that are entirely enclosed by it. Use with caution!",
    )
    parser.add_argument(
        "--add-x",
        type=int,
        dest="dx",
        metavar="x-offset",
        default=0,
        help="Add a positive or negative offset to the x coordinates of filtered lines.",
    )
    parser.add_argument(
        "--add-y",
        type=int,
        dest="dy",
        metavar="y-offset",
        default=0,
        help="Add a positive or negative offset to the y coordinates of filtered lines.",
    )
    parser.add_argument(
        "--add-z",
        type=int,
        dest="dz",
        metavar="z-offset",
        default=0,
        help="Add a positive or negative offset to the z coordinates of filtered lines (for 3d maps only).",
    )
    parser.add_argument(
        "--force-3d",
        action="store_true",
        help='Force the given map file to be parsed as 3d, also requiring 3d ranges. Only necessary if you are filtering a map fragment which does not include the "dmode 3d" line. Usage with 2d map data is subject to major breakage!',
    )
    dimgroup = parser.add_mutually_exclusive_group(required=True)
    dimgroup.add_argument(
        "--2d",
        type=int,
        nargs=4,
        dest="range",
        metavar=("minx", "maxx", "miny", "maxy"),
        help="the enclosing range to filter (for a 2d map)",
    )
    dimgroup.add_argument(
        "--3d",
        type=int,
        nargs=6,
        dest="range",
        metavar=("minx", "maxx", "miny", "maxy", "minz", "maxz"),
        help="the enclosing range to filter (for a 3d map)",
    )
    args = parser.parse_args()
    lines = []
    try:
        with args.infile.open("r", encoding="mbcs") as infile:
            lines.extend(infile.read().splitlines())
    except FileNotFoundError:
        parser.error(f"File not found: {args.infile}")
    except Exception as exc:
        parser.error(f"Could not open input file: {exc}")
    try:
        dimensions, mapdata = parse_map(lines, args.force_3d)
        map, filtered = splitfilter(dimensions, mapdata, args.range, args.enclose)
        if len(filtered) < 1:
            parser.exit(
                "No lines could be filtered in the given range. Map is unchanged."
            )
        print(f"Split map into {len(map)} map lines and {len(filtered)} filtered lines")
        if (args.dx, args.dy, args.dz) != (0, 0, 0):
            # specified offsets. Apply those to filtered data.
            delta = [args.dx, args.dy]
            if dimensions == dimensions.DIM_3d:
                delta.append(args.dz)
            deltaed = deltacoord(dimensions, filtered, delta)
            filtered.clear()
            filtered.extend(deltaed)
            deltastring = ", ".join(str(i) for i in delta)
            print(f"Offset filtered data by ({deltastring})")
        try:
            with args.outmap.open("w", encoding="mbcs") as outmap:
                outmap.write("\n".join(str(i) for i in map))
        except Exception as exc:
            parser.error(f"Failed to write map output file: {exc}")
        try:
            with args.outfiltered.open("w", encoding="mbcs") as outfiltered:
                outfiltered.write("\n".join(str(i) for i in filtered))
        except Exception as exc:
            parser.error(f"Failed to write filtered output file: {exc}")
    except ValueError as ve:
        parser.error(ve)
    except Exception as exc:
        parser.error(exc)
    if interactive:
        input("Press enter to quit')")


def yesno(text: str) -> bool:
    """Prompt for yes/no input and convert to boolean

    Args:
        text (str): The prompt to display

    Raises:
        ValueError: If the input is not a valid yes/no answer.

    Returns:
        bool: True if the user specified yes, False if they specified no.
    """
    inp = input(f"{text}\n(Y/N)>").lower()
    if inp in ("y", "yes", "t", "true", "1"):
        return True
    elif inp in ("n", "no", "f", "false", "0"):
        return False
    else:
        raise ValueError("Invalid yes/no input.")


def prompt(text: str, promptline: str = "") -> str:
    """Prompt for input with a specified message and prompt line.

    Args:
        text (str): The message to display, the full thing being prompted for.
        promptline (str): The prompt line preceding the user's input, excluding the trailing '>'.

    Returns:
        str: The text the user entered.
    """
    print(text)
    return input(f"{promptline}>")


def multiprompt(text: str, promptlines: list[str]) -> list[str]:
    """Prompt for multiple lines of input with a specified message and list of prompt lines.

    Args:
        text (str): The message to display, the full thing being prompted for.
        promptlines (list[str]): The list of prompt lines preceding the user's input, excluding the trailig '>'.

    Returns:
        list[str]: The text the user entered at all of these prompts.
    """
    print(text)
    return [input(f"{promptline}>") for promptline in promptlines]


def interact():
    print(f"CoordSlice version {version}, by x0")
    print("Interactive mode\n")
    try:
        inmap = prompt(
            "Enter the path of the map file to filter (including its extension).",
        ).replace('"', "")
        inpath = Path(inmap)
        if not inpath.exists():
            raise ValueError("Map file does not exist: " + str(inpath))
        is_3d = yesno("Is this a 3d map?")
        rangeprompts = ["min x", "max x", "min y", "max y"]
        if is_3d:
            rangeprompts.extend(["min z", "max z"])
        range = multiprompt(
            "Enter the filter range (one number at a time)", rangeprompts
        )
        _ = [int(i) for i in range]  # ensure these are all valid integers
        overlap = yesno(
            "Should the filter include anything that overlaps the range? If not, only objects fully enclosed will be included.\nCaution: Should almost always be no!"
        )
        newargs = []
        newargs.append("--3d" if is_3d else "--2d")
        newargs.extend(range)
        if overlap:
            newargs.append("--overlap")
        useoffset = yesno("Would you like to move the objects in this range?")
        if useoffset:
            offsetprompts = ["x-offset", "y-offset"]
            if is_3d:
                offsetprompts.append("z-offset")
            offsets = multiprompt(
                "Enter the offsets for each axis (one number at a time)", offsetprompts
            )
            offseti = [
                int(i) for i in offsets
            ]  # ensure these are all valid integers too
            if offseti[0] != 0:
                newargs.extend(("--add-x", offsets[0]))
            if offseti[1] != 0:
                newargs.extend(("--add-y", offsets[1]))
            if is_3d and offseti[2] != 0:
                newargs.extend(("--add-z", offsets[2]))
        outmap = prompt(
            "Enter the file name (including the extension) where everything except objects in the specified range should be stored",
        ).replace('"', "")
        outfiltered = prompt(
            "Enter the file name (including the extension) where everything in the specified range should be stored",
        ).replace('"', "")
        newargs.extend((inmap, outmap, outfiltered))
        sys.argv.extend(newargs)
    except ValueError as ve:
        if ve.args[0].startswith("invalid literal for int()"):
            raise ValueError(
                ve.args[0].replace(
                    "invalid literal for int() with base 10: ", "invalid number: "
                )
            )
        else:
            raise ve
    except KeyboardInterrupt:
        print("QUIT")
        sys.exit(0)


if __name__ == "__main__":
    main()
