# parser building blocks to handle loading different map objects

import enum
import re


class dimensions(enum.IntEnum):
    DIM_2d = 0
    DIM_3d = 1


class loctype(enum.IntEnum):
    POINT = 0
    RANGE = 1
    RANGE2 = 2  # RANGE2 is the case where only one vertical coord is present, as with platforms


# the below tuples contain a 2d and a 3d variant, matched up to values of dimensions
# These are only fragments of the full regex for any given map object,
# done like this so they may be substituted in,
# as all map objects use one of these variants to specify locations.

pointre = (r"(?P<x>-?\d+)\s+(?P<y>-?\d+)", r"(?P<x>-?\d+)\s+(?P<y>-?\d+)\s+(?P<z>-?\d+)")

rangere = (
    r"(?P<minx>-?\d+)\s+(?P<maxx>-?\d+)\s+(?P<miny>-?\d+)\s+(?P<maxy>-?\d+)",
    r"(?P<minx>-?\d+)\s+(?P<maxx>-?\d+)\s+(?P<miny>-?\d+)\s+(?P<maxy>-?\d+)\s+(?P<minz>-?\d+)\s+(?P<maxz>-?\d+)",
)

rangere2 = (
    r"(?P<minx>-?\d+)\s+(?P<maxx>-?\d+)\s+(?P<y>-?\d+)",
    r"(?P<minx>-?\d+)\s+(?P<maxx>-?\d+)\s+(?P<miny>-?\d+)\s+(?P<maxy>-?\d+)\s+(?P<z>-?\d+)",
)

# map loctype to the corresponding fragment
# this allows you to say locres[loctype][dimension] and get a valid fragment.
locres = (pointre, rangere, rangere2)


def compile_dimdupe(expr: str, loctype: loctype) -> list[re.Pattern]:
    """Creates 2/3d variations of a given regex via fragment substitution

    Args:
        expr (str): The regex, containing a single %s, to substitute the dimensional variations into
        loctype (loctype): The location type, mapped to locres.

    Returns:
        list[re.Pattern]: Compiled regexes for both 2d and 3d matches
    """
    return [re.compile(expr % i) for i in locres[loctype]]


# internal mapping of preambles to loctype and regex Patterns
exprs = {}


def make_exprs(preambles: list, loctype: loctype, postfix: bool):
    """Maps the given preambles to generated regex objects used to parse them from map data

    Args:
        preambles (list[str]): The map object preambles that match this loctype and value of postfix.
        loctype (loctype): Enumerated value of location type, point, range or range2 depending on syntax of the map.
        postfix (bool): Whether extra data is expected to come after the location declaration. A .* capture is added if True.
    """
    for preamble in preambles:
        if postfix:
            exprs[preamble] = (
                loctype,
                compile_dimdupe(rf"^(?P<pre>{preamble}\s+)%s(?P<post>\s+.*)$", loctype),
            )
        else:
            exprs[preamble] = (
                loctype,
                compile_dimdupe(rf"^(?P<pre>{preamble}\s+)%s(?P<post>)$", loctype),
            )


# use the above utility functions to generate loaders for
# all SBYW map objects with location at the time of writing
make_exprs(
    [
        "sign",
        "txt",
        "poi",
        "passage",
        "xpassage",
        "door",
        "travelpoint",
        "travelpoint3",
    ],
    loctype.POINT,
    True,
)
# checkpoint is singled out because
# it may or may not have anything coming after the point,
# and if it doesn't using postfix will cause it to fail to match due to \s+
exprs["checkpoint"] = (
    loctype.POINT,
    compile_dimdupe(r"^(?P<pre>checkpoint\s+)%s(?P<post>.*)$", loctype.POINT),
)
make_exprs(["chat", "nohealthitems", "hazard"], loctype.RANGE, False)
make_exprs(
    [
        "zone",
        "premium_zone",
        "belt",
        "mplatform",
        "mhazard",
        "wall",
        "staircase",
        "sound_source",
        "url_source",
        "ambience",
        "url_ambience",
        "xsource",
        "xambience",
        "bounded_source",
        "xdoor",
        "story",
        "mpoi",
        "travelplace",
        "hookblock",
        "obscure",
    ],
    loctype.RANGE,
    True,
)
make_exprs(["platform", "vanishing"], loctype.RANGE2, True)
