# functions to manipulate the map data

from .exprs import dimensions, loctype
from .objects import point2d, point3d, range2d, range3d, map_object2d, map_object3d


def splitfilter(
    dim: dimensions, lines: list, range: list[int], enclose: bool = True
) -> tuple[list[str], list[str]]:
    """Split a list of map data based on a coordinate range.

    Args:
        dim (dimensions): The dimensions of the map to split.
        lines (list): The map data, a list of a combination of str and map_object* objects.
        range (list[int]): The range to filter with, must be exactly 4 or 6 elements depending on dimensions.
        enclose (bool, optional): Whether to filter objects that are contained entirely within the range (True) or that merely intersect it (False).. Defaults to True.

    Raises:
        ValueError: The given range does not match the given dimensions.

    Returns:
        tuple[list[str], list[str]]: A pair of lists, where the first is the non-filtered data and the second is the filtered data.
    """
    if len(range) == 6 and dim == dimensions.DIM_2d:
        raise ValueError("map is 2d but a 3d range was given")
    elif len(range) == 4 and dim == dimensions.DIM_3d:
        raise ValueError("map is 3d but a 2d range was given")
    ret = (list(), list())
    enclosure = range3d(*range) if len(range) == 6 else range2d(*range)
    for line in lines:
        dest = 0
        if dim == dimensions.DIM_2d and isinstance(line, map_object2d):
            dest = 1 if line.contained(enclosure, enclose) else 0
        elif dim == dimensions.DIM_3d and isinstance(line, map_object3d):
            dest = 1 if line.contained(enclosure, enclose) else 0
        else:
            dest = 0
        ret[dest].append(line)
    return ret


def deltacoord(dim: dimensions, lines: list, delta: list[int]) -> list:
    """Offset the given map objects by a constant along each axis.

    Args:
        dim (dimensions): The dimensions of the map data.
        lines (list): The map data, a list of a combination of str and map_object* objects.
        delta (list[int]): The amount to offset per-axis. Must have length of 2 or 3 depending on dimensions.

    Raises:
        ValueError: Given delta does not match given dimensions.

    Returns:
        list: The map data after the delta was applied to any objects.
    """
    ret = []
    if dim == dimensions.DIM_2d:
        if len(delta) == 3:
            raise ValueError("map is 2d but a 3d delta was given")
        dx, dy = delta[0], delta[1]
        for line in lines:
            if isinstance(line, map_object2d):
                oldloc = line.location
                newloc = None
                if line.loctype == loctype.POINT:
                    newloc = point2d(oldloc.x + dx, oldloc.y + dy, oldloc.match)
                else:
                    newloc = range2d(
                        oldloc.minx + dx,
                        oldloc.maxx + dx,
                        oldloc.miny + dy,
                        oldloc.maxy + dy,
                        oldloc.vdupe,
                        oldloc.match,
                    )
                ret.append(map_object2d(line.line, line.loctype, newloc))
            else:
                ret.append(line)
    elif dim == dimensions.DIM_3d:
        if len(delta) == 2:
            raise ValueError("map is 3d but a 2d delta was given")
        dx, dy, dz = delta[0], delta[1], delta[2]
        for line in lines:
            if isinstance(line, map_object3d):
                oldloc = line.location
                newloc = None
                if line.loctype == loctype.POINT:
                    newloc = point3d(
                        oldloc.x + dx, oldloc.y + dy, oldloc.z + dz, oldloc.match
                    )
                else:
                    newloc = range3d(
                        oldloc.minx + dx,
                        oldloc.maxx + dx,
                        oldloc.miny + dy,
                        oldloc.maxy + dy,
                        oldloc.minz + dz,
                        oldloc.maxz + dz,
                        oldloc.vdupe,
                        oldloc.match,
                    )
                ret.append(map_object3d(line.line, line.loctype, newloc))
            else:
                ret.append(line)
    else:
        raise ("invalid dimension")
    return ret


__all__ = ["splitfilter", "deltacoord"]
