# object definitions and the loader.

from dataclasses import dataclass
import re  # just for re.match type

from .exprs import exprs, dimensions, loctype

"""
For the below classes,
the classmethod from_re is used to transform a match object to the corresponding dataclass,
by  extracting from named groups, int conversion, and argument passing to the constructor.
"""


@dataclass
class point2d:
    x: int
    y: int
    match: re.match

    @classmethod
    def from_re(cls, match):
        return cls(int(match["x"]), int(match["y"]), match)


@dataclass
class point3d:
    x: int
    y: int
    z: int
    match: re.match

    @classmethod
    def from_re(cls, match):
        return cls(int(match["x"]), int(match["y"]), int(match["z"]), match)


@dataclass
class range2d:
    minx: int
    maxx: int
    miny: int
    maxy: int
    vdupe: bool = None
    match: re.match = None

    @classmethod
    def from_re(cls, match):
        if "y" in match.groupdict():  # range2
            return cls(
                int(match["minx"]),
                int(match["maxx"]),
                int(match["y"]),
                int(match["y"]),
                True,
                match,
            )
        else:
            return cls(
                int(match["minx"]),
                int(match["maxx"]),
                int(match["miny"]),
                int(match["maxy"]),
                False,
                match,
            )

    def contains_point(self, point: point2d) -> bool:
        """Determines if this range contains the given point

        Args:
            point (point2d): The point to check

        Returns:
            bool: Whether the given point is inside this range.
        """
        return (
            point.x >= self.minx
            and point.x <= self.maxx
            and point.y >= self.miny
            and point.y <= self.maxy
        )

    def contains_range(self, range, enclosed: bool = True) -> bool:
        """Determines if this range contains the given range

        Args:
            range (range2d): The range to check
            enclosed (bool, optional): Require the range to be fully enclosed if True, or check for overlap if False. Defaults to True.

        Returns:
            bool: whether the given range is contained within this one
        """
        if enclosed:
            return (
                range.minx >= self.minx
                and range.maxx <= self.maxx
                and range.miny >= self.miny
                and range.maxy <= self.maxy
            )
        else:
            if (
                range.maxx < self.minx
                or range.minx > self.maxx
                or range.maxy < self.miny
                or range.miny > self.maxy
            ):
                return False
            else:
                return True


@dataclass
class range3d:
    minx: int
    maxx: int
    miny: int
    maxy: int
    minz: int
    maxz: int
    vdupe: bool = False
    match: re.match = None

    @classmethod
    def from_re(cls, match):
        if "z" in match.groupdict():
            return cls(
                int(match["minx"]),
                int(match["maxx"]),
                int(match["miny"]),
                int(match["maxy"]),
                int(match["z"]),
                int(match["z"]),
                True,
                match,
            )
        else:
            return cls(
                int(match["minx"]),
                int(match["maxx"]),
                int(match["miny"]),
                int(match["maxy"]),
                int(match["minz"]),
                int(match["maxz"]),
                False,
                match,
            )

    def contains_point(self, point: point3d):
        """Determines if this range contains the given point

        Args:
            point (point3d): The point to check

        Returns:
            bool: Whether the given point is inside this range.
        """
        return (
            point.x >= self.minx
            and point.x <= self.maxx
            and point.y >= self.miny
            and point.y <= self.maxy
            and point.z >= self.minz
            and point.z <= self.maxz
        )

    def contains_range(self, range, enclosed: bool = True):
        """Determines if this range contains the given range

        Args:
            range (range3d): The range to check
            enclosed (bool, optional): Require the range to be fully enclosed if True, or check for overlap if False. Defaults to True.

        Returns:
            bool: whether the given range is contained within this one
        """
        if enclosed:
            return (
                range.minx >= self.minx
                and range.maxx <= self.maxx
                and range.miny >= self.miny
                and range.maxy <= self.maxy
                and range.minz >= self.minz
                and range.maxz <= self.maxz
            )
        else:
            if (
                range.maxx < self.minx
                or range.minx > self.maxx
                or range.maxy < self.miny
                or range.miny > self.maxy
                or range.maxz < self.minz
                or range.minz > self.maxz
            ):
                return False
            else:
                return True


@dataclass
class map_object2d:
    line: str
    loctype: loctype
    location: point2d | range2d

    def __str__(self):
        match = self.location.match
        gd = match.groupdict(default="")
        pre = gd["pre"]
        post = gd["post"]
        if isinstance(self.location, point2d):
            x, y = self.location.x, self.location.y
            return f"{pre}{x} {y}{post}"
        elif isinstance(self.location, range2d):
            minx, maxx = self.location.minx, self.location.maxx
            miny, maxy = self.location.miny, self.location.maxy
            if self.location.vdupe:
                return f"{pre}{minx} {maxx} {miny}{post}"
            else:
                return f"{pre}{minx} {maxx} {miny} {maxy}{post}"
        else:
            return self.line

    def contained(self, range: range2d, enclosed: bool = True) -> bool:
        """Call a check on the given range to determine if this object is contained within it

        Args:
            range (range2d): The range specified for filtering
            enclosed (bool, optional): Require this object to be completely enclosed within the range if True, or overlap it if False. Defaults to True.

        Returns:
            bool: Whether this object is inside the given range.
        """
        if self.loctype == loctype.POINT:
            return range.contains_point(self.location)
        else:
            return range.contains_range(self.location, enclosed)


@dataclass
class map_object3d:
    line: str
    loctype: loctype
    location: point3d | range3d

    def __str__(self):
        match = self.location.match
        gd = match.groupdict(default="")
        pre = gd["pre"]
        post = gd["post"]
        if isinstance(self.location, point3d):
            x, y, z = self.location.x, self.location.y, self.location.z
            return f"{pre}{x} {y} {z}{post}"
        elif isinstance(self.location, range3d):
            minx, maxx = self.location.minx, self.location.maxx
            miny, maxy = self.location.miny, self.location.maxy
            minz, maxz = self.location.minz, self.location.maxz
            if self.location.vdupe:
                return f"{pre}{minx} {maxx} {miny} {maxy} {minz}{post}"
            else:
                return f"{pre}{minx} {maxx} {miny} {maxy} {minz} {maxz}{post}"
        else:
            return self.line

    def contained(self, range: range3d, enclosed: bool = True) -> bool:
        """Call a check on the given range to determine if this object is contained within it

        Args:
            range (range3d): The range specified for filtering
            enclosed (bool, optional): Require this object to be completely enclosed within the range if True, or overlap it if False. Defaults to True.

        Returns:
            bool: Whether this object is inside the given range.
        """
        if self.loctype == loctype.POINT:
            return range.contains_point(self.location)
        else:
            return range.contains_range(self.location, enclosed)


# map the current map's dimensions and the loctype of a given object
# to the corresponding map object and location type
mapping = {
    (dimensions.DIM_2d, loctype.POINT): (map_object2d, point2d),
    (dimensions.DIM_2d, loctype.RANGE): (map_object2d, range2d),
    (dimensions.DIM_2d, loctype.RANGE2): (map_object2d, range2d),
    (dimensions.DIM_3d, loctype.POINT): (map_object3d, point3d),
    (dimensions.DIM_3d, loctype.RANGE): (map_object3d, range3d),
    (dimensions.DIM_3d, loctype.RANGE2): (map_object3d, range3d),
}


def parse_line(line: str, dim: dimensions) -> map_object2d | map_object3d:
    """Parses a map line and returns a populated map object if possible

    Args:
        line (str): The map line to parse
        dim (dimensions): Dimensions of the map. Determines how coordinates are represented.

    Returns:
        map_object2d | map_object3d: The parsed map object corresponding to the given line and dimensions.
    """
    s = line.lstrip().split()
    preamble = s[0]
    if preamble in exprs:
        loctype, res = exprs[preamble]
        if m := res[dim].fullmatch(line):
            mapobj, location = mapping[(dim, loctype)]
            return mapobj(line, loctype, location.from_re(m))


def parse_map(lines: list[str], force_3d: bool = False) -> tuple[dimensions, list]:
    """Parses a map and gives a list of a combination of lines and map objects

    Args:
        lines (list[str]): The list of map lines (no trailing newline characters). As from splitlines.
        force_3d (bool): Whether to force interpreting data as 3d, even without a dmode line.


    Returns:
        tuple[dimensions, list]: A dimension marker and a list, containing map objects for all parsable lines and strings for everything else
    """
    dimension = dimensions.DIM_3d if force_3d else dimensions.DIM_2d
    ret = []
    for line in lines:
        if len(line.strip()) == 0:
            ret.append(line)
            continue
        if line.startswith("dmode"):
            s = line.split()
            if s[1] == "3d":
                dimension = dimensions.DIM_3d
            ret.append(line)
        elif obj := parse_line(line, dimension):
            ret.append(obj)
        else:
            ret.append(line)
    return (dimension, ret)
